<?php

namespace App\Models;

use CodeIgniter\Model;

class ContactMessageModel extends Model
{
    protected $table = 'contacts';
    protected $primaryKey = 'id';
    protected $allowedFields = ['name', 'email', 'phone', 'subject', 'message', 'is_read', 'folder', 'created_at', 'updated_at',];
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';

    protected $validationRules = [
        'name' => 'required|min_length[3]|max_length[255]',
        'email' => 'required|valid_email|max_length[255]',
        'phone' => 'required|regex_match[/^[0-9()+-]{10,20}$/]',
        'subject' => 'required|min_length[5]|max_length[255]',
        'message' => 'required|min_length[10]',
    ];

    protected $validationMessages = [
        'name' => [
            'required' => 'برجاء ادخال الاسم',
            'min_length' => 'برجاء ادخال الاسم كحد أدنى {param} أحرف',
            'max_length' => 'برجاء ادخال الاسم كحد أقصى {param} أحرف',
        ],
        'email' => [
            'required' => 'برجاء ادخال البريد الالكتروني',
            'valid_email' => 'برجاء ادخال بريد الكتروني صحيح',
            'max_length' => 'برجاء ادخال البريد الالكتروني كحد أقصى {param} أحرف',
        ],
        'phone' => [
            'required' => 'برجاء ادخال رقم الهاتف',
            'regex_match' => 'برجاء ادخال رقم هاتف صحيح',
        ],
        'subject' => [
            'required' => 'برجاء ادخال العنوان',
            'min_length' => 'برجاء ادخال عنوان يحتوي كحد أدنى {param} أحرف',
            'max_length' => 'برجاء ادخال عنوان يحتوي كحد أقصى {param} أحرف',
        ],
        'message' => [
            'required' => 'برجاء ادخال الرسالة',
            'min_length' => 'برجاء ادخال رسالة تحتوي كحد أدنى {param} أحرف',
        ],
    ];

    public function getMessagesByFolder($folder)
    {
        return $this->where('folder', $folder)->orderBy('created_at', 'desc')->findAll();
    }

    public function getMessageById($id)
    {
        return $this->find($id);
    }

    public function markAsRead($id, $is_read)
    {
        $this->update($id, ['is_read' => $is_read]);
    }

    public function markAsUnread($id)
    {
        $data = ['is_read' => 0];
        $this->update($id, $data);
    }

    public function moveToFolder($id, $folder)
    {
        $this->update($id, ['folder' => $folder]);
    }

    public function insertMessage($data)
    {
        $this->insert($data);
    }

    public function insertReply($data)
    {
        $builder = $this->db->table('contacts');
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['updated_at'] = date('Y-m-d H:i:s');
        $builder->insert($data);
        $this->moveToFolder($this->db->insertID(), 'sent');
    }

    public function delete($id = null, bool $purge = false, array $options = [])
    {
        if ($id !== null) {
            $this->builder()->where('id', $id);
        } else if ($purge) {
            $this->builder()->where('folder', 'trash');
        } else {
            $this->builder()->update(['folder' => 'trash']);
        }
        return $this->builder()->delete();
    }

}